
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cassert>


#if !defined( _UTILS_HASH_AND_LIST ) 
#define  _UTILS_HASH_AND_LIST 

template <class Type>
class HashAndListItem
{
private:
    Type            *datum;
    HashAndListItem *previous,
                    *next;
    int              posInHash;

public:
    HashAndListItem( Type * datum )
    {
        this->previous = NULL;
        this->datum = datum;
        this->next = NULL;
        this->posInHash = -1;
    }
    HashAndListItem( HashAndListItem *previous, Type * datum, HashAndListItem *next )
    {
        this->previous = previous;
        this->datum = datum;
        this->next = next;
        this->posInHash = -1;

        if ( previous != NULL ) previous->next = this;
        if ( next     != NULL ) next->previous = this;
    }
    ~HashAndListItem()
    {
        // DO NOTHING. The datum must be deallocated outside this class.
    }

    inline Type * get() { return datum; }

    inline HashAndListItem * getPrevious() { return previous; }
    inline HashAndListItem * getNext() { return next; }

    inline void set( Type * datum ) { this->datum = datum; }
    inline void setPrevious( HashAndListItem * ptr ) { this->previous = ptr; }
    inline void setNext( HashAndListItem * ptr ) { this->next = ptr; }

    inline void setPosInHash( int pos ) { this->posInHash = pos; }
    inline int  getPosInHash() { return this->posInHash; }
};

template <class Type>
class HashAndList
{
private:
    HashAndListItem<Type>   *first,
                            *last;
    HashAndListItem<Type>  **vector;
    int                      size, S, T;
    int                      colisionsCounter;

    void rehash()
    {
        delete [] vector;
        
        T = T << 1;
        vector = new HashAndListItem<Type> * [T];
        for( int i=0; i < T; i++ ) vector[i] = NULL;
        size=0;

        S = T/5;
        while( S < T && (T%S) == 0 ) S++;

        for( HashAndListItem<Type> *item = first; item != NULL; item = item->getNext() ) {
            insert( item );
        }
    }
    void clearList()
    {
        while( first != NULL ) {
            HashAndListItem<Type>   *temp = first;
            first = first->getNext();
            delete temp;
        }
        first = last = NULL;
    }
    void removeFromList( HashAndListItem<Type> * item )
    {
        HashAndListItem<Type> * previous = item->getPrevious();
        HashAndListItem<Type> * next     = item->getNext();

        if ( item == first ) {

            first = item->getNext();
            if ( first != NULL ) first->setPrevious( NULL );

        } else if ( item == last ) {

            last = item->getPrevious();
            if ( last != NULL ) last->setNext( NULL );

        } else {
            if ( previous != NULL ) previous->setNext( next );
            if (     next != NULL ) next->setPrevious( previous );
        }

        delete item;
    }

public:


    HashAndList( int initialSize=0 )
    {
        T = ( initialSize > 0 ) ? initialSize : (1 << 18);

        vector = new HashAndListItem<Type> * [T];
        first = last = NULL;

        S = T/5;
        while( S < T && (T%S) == 0 ) S++;

        clear( true );
    }
    ~HashAndList()
    {
        delete [] vector;

        clearList();
    }
    int getColisions() { return colisionsCounter; }
    int getSize() { return size; }

    void clear( bool doMemset )
    {
        size=0;
        colisionsCounter=0;

        if ( doMemset ) {
            for( int i=0; i < T; i++ ) vector[i] = NULL;
        }

        clearList();
    }


    void insert( Type * datum )
    {
        int pos = find( datum );

        if ( vector[pos] == NULL ) {

            HashAndListItem<Type> * item = new HashAndListItem<Type>( last, datum, NULL );
            last = item;
            if ( 0 == size ) { first = item; }

            vector[pos] = item;
            item->setPosInHash( pos );

            ++size;

        } else {
            // DO NOTHING IF THE datum ALREADY EXISTS
        }
    }
    int find( Type * datum )
    {
        int pos = datum->getHashCode() % T;

        while( vector[pos] != NULL && !datum->equals( vector[pos]->get() ) ) {

            pos = (pos+S)%T;
        }
        return pos;
    }

    Type * get( Type * datum )
    {
        int pos = find(datum);
        return (vector[pos] != NULL ) ? vector[pos]->get() : NULL;
    }

    void remove( int pos )
    {
        if ( size > 0 ) {
            HashAndListItem<Type> * item = vector[pos];
            vector[pos] = NULL;
            --size;

            removeFromList( item );
        }
    }


    class iterator {
        private:
            int index;
            HashAndListItem<Type> * currentItem;

        public:
            iterator( int pos=0, HashAndListItem<Type> *item=NULL ) : index(pos), currentItem(item) {}

            iterator( const iterator & iter ) : index(iter.index), currentItem(iter.currentItem) {}

            // Prefix operators
            iterator & operator++() { ++index; currentItem=currentItem->getNext();     return *this; }
            iterator & operator--()
            {
                --index;
                currentItem = (index == size) ? last : currentItem->getPrevious();
                return *this;
            }
/*
    Suffix operators
            iterator   operator++(int) { iterator temp(*this); ++(*this); return temp; }
            iterator   operator--(int) { iterator temp(*this); --(*this); return temp; }
*/
            bool operator!=( iterator  it ) { return index != it.index; }
            bool operator<=( iterator  it ) { return index <= it.index; }
            bool operator<(  iterator  it ) { return index <  it.index; }

            Type * operator*() { return currentItem->get(); }
    };

    iterator begin() { return iterator( 0, first ); }
    iterator end()   { return iterator( size, NULL ); }
};

#endif // _UTILS_HASH_AND_LIST 

#include <vector>
#include <algorithm>

class Word
{
private:
    static const int HASH_MAX = (1 << 28);

    char       *word;
    int         counter;
    long long   hash;

    int         posInHash;
    int         posInHeap;

    Word       *next; // For the queue

public:

    Word( const char *word, int counter )
    {
        this->counter = counter;

        int L = strlen(word);

        this->word = new char [ L+1 ];
        this->word[L] = '\0';

        hash = 0;
        for( int i=L-1; i >= 0; i-- ) {
            this->word[i] = word[i];

            hash = ( ((hash * 31) % HASH_MAX) + (word[i]-' ') ) % HASH_MAX;
        }
        assert( hash > 0 );
        assert( hash < (1LL << 31) );
    }
    ~Word()
    {
        delete [] word;
    }

    static int hashCode( const char *token )
    {
        long hash = 0;
        for( int i=strlen(token)-1; i >= 0; i-- ) {
            hash = ( ((hash * 31) % HASH_MAX) + (token[i]-' ') ) % HASH_MAX;
        }
        assert( hash > 0 );
        assert( hash < (1LL << 31) );

        return (int)hash;
    }

    inline void increaseCounter() { ++counter; }
    inline void decreaseCounter() { --counter; }
    inline int  getCounter() { return counter; }

    inline int hashCode() { return hash; }
    inline int getHashCode() { return hash; }

    inline char * getWord() { return word; }

    inline int  getPosInHash() { return posInHash; }
    inline int  getPosInHeap() { return posInHeap; }
    inline void setPosInHash( int pos ) { posInHash = pos; }
    inline void setPosInHeap( int pos ) { posInHeap = pos; }

    inline bool equals( Word *other )
    {
        return (this == other || 0 == strcmp( this->word, other->word ));
    }
    inline bool equals( const char *token )
    {
        return (0 == strcmp( this->word, token ));
    }
    int compareTo( Word *other )
    {
        if ( this->counter > other->counter ) return  -1;
        if ( this->counter < other->counter ) return   1;
        return strcmp( this->word, other->word );
    }
    static char * strdup( const char *s )
    {
        char *t = new char [ strlen(s)+1 ];

        strcpy( t, s );

        return t;
    }

    inline Word * getNext() { return next; }
    inline void setNext( Word * next ) { this->next = next; }
};

struct compare_words {
    bool operator() ( Word *x, Word *y ) { return x->compareTo(y) < 0; }
};



class Day
{
private:
    char   **words;
    int      size;
    int      reserved;
    Day     *next;

public:
    
    Day()
    {
        reserved = 1<<14;
        size = 0;
        words = new char * [reserved];
        next = NULL;
    }
    ~Day()
    {
        while( --size >= 0 ) delete [] words[size];
        delete [] words;
    }

    void add( const char * word )
    {
        words[size++] = Word::strdup( word );

        if ( size == reserved ) {
            reserved <<= 1;
            char ** temp = new char * [ reserved ];
            memcpy( temp, words, size * sizeof( char * ) );
            delete [] words;
            words = temp;
        }
    }

    inline const char *getWord( int pos ) { return words[pos]; }
    inline int getSize() { return size; }

    inline Day * getNext() { return next; }
    inline void setNext( Day *next ) { this->next = next; }
};

template <class T> class Queue
{
private:
    T       *first;
    T       *last;
    int      size;

public:
    Queue<T>()
    {
        first = last = NULL;
        size = 0;
    }
    ~Queue<T>()
    {
        T *temporary;
        while( first != NULL ) {
            temporary = first;
            first = first->getNext();
            delete temporary;
        }
    }

    inline int getSize() { return size; }

    void add( T *datum )
    {
        if ( 0 == size ) {
            datum->setNext(NULL);
            first = last = datum;
            size = 1;
        } else {
            datum->setNext(NULL);
            last->setNext( datum );
            last = datum;
            ++size;
        }
    }

    T * remove()
    {
        if ( 0 == size ) {
            fprintf( stderr, "Empty queue of days!\n" ); assert( 0 );
        }

        T *temporary = first;
        first = first->getNext();
        temporary->setNext(NULL);
        --size;

        return temporary;
    }
};

class TrendingTopic
{
private:
    HashAndList<Word>   currentWords;
    Queue<Day>          availableDays;
    Day                *currentDay;
    Day                *oldestDay;

    char                buffer[4096];


public:
    TrendingTopic()
    {
        currentDay = NULL;
        oldestDay = NULL;
    }
    ~TrendingTopic()
    {
        for( HashAndList<Word>::iterator it = currentWords.begin(); it < currentWords.end(); ++it ) {
            delete *it;
        }
    }

    void processInput( FILE *input )
    {
        buffer[0] = '\0';
        while( fscanf( input, "%s", buffer ) == 1 ) {

            if ( 0 == strcmp( buffer, "<text>" ) ) {
                loadNewDay( input );
            } else if ( 0 == strcmp( buffer, "<top" ) ) {
                int N;
                fscanf( input, "%d", &N );
                fscanf( input, "%s", buffer );
                showMostFrequentWords( N );
            } else {
                fprintf( stderr, "FATAL ERROR if this line is reached: |%s|\n", buffer );
            }
        }
    }
    void loadNewDay( FILE *input )
    {
        currentDay = new Day();
        while( fscanf( input, "%s", buffer ) == 1 ) {
            if ( 0 == strcmp( buffer, "</text>" ) ) break;
            if ( strlen(buffer) >= 4 ) currentDay->add( buffer );
        }
        availableDays.add( currentDay );

        for( int i=0; i < currentDay->getSize(); i++ ) {
            Word *word = new Word( currentDay->getWord(i), 1 );
            Word *wordPtr = currentWords.get( word );

            if ( wordPtr == NULL ) {
                currentWords.insert( word );
            } else {
                wordPtr->increaseCounter();
                delete word;
            }
        }

        if ( availableDays.getSize() > 7 ) {

            oldestDay = availableDays.remove();

            for( int i=0; i < oldestDay->getSize(); i++ ) {
                Word word( oldestDay->getWord(i), 1 );

                Word *wordPtr = currentWords.get( &word );
                assert( wordPtr != NULL );
                wordPtr->decreaseCounter();
            }

            delete oldestDay;
        }
    }
    void showMostFrequentWords( int N )
    {
        Word * mostFrequents[N+1];
        memset( mostFrequents, 0, (N+1)*sizeof(Word *) );

        int n=0;

        for( HashAndList<Word>::iterator it = currentWords.begin(); it < currentWords.end(); ++it ) {

            Word *word = *it;

            int i = n;
            while( i > 0  &&  word->getCounter() > mostFrequents[i-1]->getCounter() ) {
                mostFrequents[i] = mostFrequents[i-1];
                --i;
            }
            mostFrequents[i] = word;
            if ( n < N ) ++n;
        }
        int counter = mostFrequents[N-1]->getCounter();

        std::vector<Word *>  listToBeShown;

        for( HashAndList<Word>::iterator it = currentWords.begin(); it < currentWords.end(); ++it ) {

            Word *word = *it;

            if ( word->getCounter() >= counter ) listToBeShown.push_back( word );
        }

        sort( listToBeShown.begin(), listToBeShown.end(), compare_words() );

        fprintf( stdout, "<top %d>\n", N );
        for( std::vector<Word *>::iterator it = listToBeShown.begin(); it < listToBeShown.end(); ++it ) {
            fprintf( stdout, "%s %d\n", (*it)->getWord(), (*it)->getCounter() );
        }
        fprintf( stdout, "</top>\n" );
    }
};



int main( int argc, char *argv[] )
{
    TrendingTopic tt;

    tt.processInput( stdin );

    return EXIT_SUCCESS;
}
