
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cassert>

class Word
{
private:
    static const int HASH_MAX = (1 << (31-5)); // Because we use 31 (2^5-1) in the computation of the hash code

    char       *word;
    int         counter;
    long long   hash;

    int         posInHash;
    int         posInHeap;

    Word       *next; // For the queue

public:

    Word( const char *word, int counter )
    {
        this->counter = counter;

        int L = strlen(word);

        this->word = new char [ L+1 ];
        this->word[L] = '\0';

        hash = 0;
        for( int i=L-1; i >= 0; i-- ) {
            this->word[i] = word[i];

            hash = ( ((hash * 31) % HASH_MAX) + (word[i]-' ') ) % HASH_MAX;
        }
        assert( hash > 0 );
        assert( hash < (1LL << 31) );
    }
    ~Word()
    {
        delete [] word;
    }

    static int hashCode( const char *token )
    {
        long hash = 0;
        for( int i=strlen(token)-1; i >= 0; i-- ) {
            hash = ( ((hash * 31) % HASH_MAX) + (token[i]-' ') ) % HASH_MAX;
        }
        assert( hash > 0 );
        assert( hash < (1LL << 31) );

        return (int)hash;
    }

    inline void increaseCounter() { ++counter; }
    inline void decreaseCounter() { --counter; }
    inline int  getCounter() { return counter; }

    inline int hashCode() { return hash; }

    inline char * getWord() { return word; }

    inline int  getPosInHash() { return posInHash; }
    inline int  getPosInHeap() { return posInHeap; }
    inline void setPosInHash( int pos ) { posInHash = pos; }
    inline void setPosInHeap( int pos ) { posInHeap = pos; }

    inline bool equals( Word *other )
    {
        return (this == other || 0 == strcmp( this->word, other->word ));
    }
    inline bool equals( const char *token )
    {
        return (0 == strcmp( this->word, token ));
    }
    int compareTo( Word *other )
    {
        if ( this->counter > other->counter ) return  -1;
        if ( this->counter < other->counter ) return   1;
        return strcmp( this->word, other->word );
    }
    static char * strdup( const char *s )
    {
        char *t = new char [ strlen(s)+1 ];

        strcpy( t, s );

        return t;
    }

    inline Word * getNext() { return next; }
    inline void setNext( Word * next ) { this->next = next; }
};

class HashAndHeap
{
private:
    Word    **heap;
    int       heapMaxSize;

    Word    **hash;
    int       hashMaxSize;

    int       size;

public:
    HashAndHeap()
    {
        heapMaxSize = 1 << 14;
        heap = new Word * [heapMaxSize+1];

        memset( heap, 0, (heapMaxSize+1)*sizeof( Word * ) );

        hashMaxSize = 1 << 14;
        hash = new Word * [hashMaxSize];

        memset( hash, 0, hashMaxSize*sizeof( Word * ) );

        size=0;
    }
    ~HashAndHeap()
    {
        for( int i=1; i <= size; i++ ) delete heap[i];

        delete [] heap;
        delete [] hash;
    }

    inline int getSize() { return size; }

    inline Word * top() { return heap[1]; }

    int find( int hashcode, const char *token )
    {
        // Search in hash, returns the first empty pos if the token doesn't exist
        int pos = hashcode % hashMaxSize;

        while( hash[pos] != NULL  &&  ! hash[pos]->equals( token ) ) pos = (pos+1)%hashMaxSize;

        return pos;
    }

    void push( const char *token )
    {
        int hashcode = Word::hashCode( token );
        int pos = find( hashcode, token );

        if ( NULL == hash[pos] ) {

            Word * newWord = new Word( token, 1 );
            hash[pos] = newWord;
            newWord->setPosInHash( pos );

            heap[++size] = newWord;
            newWord->setPosInHeap( size );
            increaseKey( size );

            if ( 4*size > 3*hashMaxSize ) rehash();
            if ( size == heapMaxSize ) reheap();

        } else {

            hash[pos]->increaseCounter();
            increaseKey( hash[pos]->getPosInHeap() );
        }
    }
    void pushAgain( Word *word )
    {
        // The word already will be in the same position in hash
        assert( word == hash[ word->getPosInHash() ] );

        /* In the case that words were actually removed from hash the
           following two commented lines are the correct, instead of the
           above assert().
        assert( NULL == hash[ word->getPosInHash() ] );
        hash[ word->getPosInHash() ] = word;
        */

        heap[++size] = word;
        word->setPosInHeap( size );
        increaseKey( size );
    }
    void decreaseCounter( const char *token )
    {
        int hashcode = Word::hashCode( token );
        int pos = find( hashcode, token );

        if ( 0 <= pos  &&  hash[pos]->getCounter() > 0 ) {
            hash[pos]->decreaseCounter();
            maxHeapify( hash[pos]->getPosInHeap() );
        }
    }
    void remove()
    {
        // The word is not removed from the hash. We need it for the pushAgain()
        // Word *word = heap[1];
        // hash[ word->getPosInHash() ] = NULL;

        heap[1] = heap[size--];
        maxHeapify( 1 );
    }

    void maxHeapify( int pos )
    {
        int left = pos << 1;
        int right = left+1;
        int largest = pos;

        if ( left  <= size  &&  heap[left ]->compareTo( heap[largest] ) < 0 ) largest = left;
        if ( right <= size  &&  heap[right]->compareTo( heap[largest] ) < 0 ) largest = right;

        if ( largest != pos ) {

            Word * tmp = heap[pos];
            heap[pos] = heap[largest];
            heap[largest] = tmp;

            heap[pos]->setPosInHeap( pos );
            heap[largest]->setPosInHeap( largest );

            maxHeapify( largest );
        }
    }
    void increaseKey( int pos )
    {
        while( pos > 1 ) {
            int parent = pos >> 1;

            if ( heap[pos]->compareTo( heap[parent] ) < 0 ) {

                Word * tmp = heap[pos];
                heap[pos] = heap[parent];
                heap[parent] = tmp;

                heap[pos]->setPosInHeap( pos );
                heap[parent]->setPosInHeap( parent );

                pos = parent;
            } else {
                pos = 0;
            }
        }
    }

    void rehash()
    {
        delete [] hash;
        hashMaxSize <<= 1;

        hash = new Word * [hashMaxSize];
        memset( hash, 0, hashMaxSize*sizeof( Word * ) );

        for( int i=1; i <= size; i++ ) {
            
            int pos = find( heap[i]->hashCode(), heap[i]->getWord() );

            if ( NULL == hash[pos] ) {
                hash[pos] = heap[i];
                hash[pos]->setPosInHash( pos );
            } else {
                fprintf( stderr, "Impossible to have the same word two times in hash!" ); assert( 0 );
            }
        }
    }
    void reheap()
    {
        heapMaxSize <<= 1;

        Word **newHeap = new Word * [heapMaxSize+1];

        memcpy( newHeap, heap, (size+1) * sizeof( Word * ) );

        delete [] heap;

        heap = newHeap;
    }
};

class Day
{
private:
    char   **words;
    int      size;
    int      reserved;
    Day     *next;

public:
    
    Day()
    {
        reserved = 1<<14;
        size = 0;
        words = new char * [reserved];
        next = NULL;
    }
    ~Day()
    {
        while( --size >= 0 ) delete [] words[size];
        delete [] words;
    }

    void add( const char * word )
    {
        words[size++] = Word::strdup( word );

        if ( size == reserved ) {
            reserved <<= 1;
            char ** temp = new char * [ reserved ];
            memcpy( temp, words, size * sizeof( char * ) );
            delete [] words;
            words = temp;
        }
    }

    inline const char *getWord( int pos ) { return words[pos]; }
    inline int getSize() { return size; }

    inline Day * getNext() { return next; }
    inline void setNext( Day *next ) { this->next = next; }
};

template <class T> class Queue
{
private:
    T       *first;
    T       *last;
    int      size;

public:
    Queue<T>()
    {
        first = last = NULL;
        size = 0;
    }
    ~Queue<T>()
    {
        T *temporary;
        while( first != NULL ) {
            temporary = first;
            first = first->getNext();
            delete temporary;
        }
    }

    inline int getSize() { return size; }

    void add( T *datum )
    {
        if ( 0 == size ) {
            datum->setNext(NULL);
            first = last = datum;
            size = 1;
        } else {
            datum->setNext(NULL);
            last->setNext( datum );
            last = datum;
            ++size;
        }
    }

    T * remove()
    {
        if ( 0 == size ) {
            fprintf( stderr, "Empty queue of days!\n" ); assert( 0 );
        }

        T *temporary = first;
        first = first->getNext();
        temporary->setNext(NULL);
        --size;

        return temporary;
    }
};

class TrendingTopic
{
private:
    HashAndHeap     currentWords;
    Queue<Day>      availableDays;
    Day            *currentDay;
    Day            *oldestDay;

    char            buffer[4096];


public:
    TrendingTopic()
    {
        currentDay = NULL;
        oldestDay = NULL;
    }
    ~TrendingTopic()
    {
    }

    void processInput( FILE *input )
    {
        buffer[0] = '\0';
        while( fscanf( input, "%s", buffer ) == 1 ) {

            if ( 0 == strcmp( buffer, "<text>" ) ) {
                loadNewDay( input );
            } else if ( 0 == strcmp( buffer, "<top" ) ) {
                int N;
                fscanf( input, "%d", &N );
                fscanf( input, "%s", buffer );
                showMostFrequentWords( N );
            } else {
                fprintf( stderr, "This program should never reach this line: |%s|\n", buffer );
            }
        }
    }
    void loadNewDay( FILE *input )
    {
        currentDay = new Day();
        while( fscanf( input, "%s", buffer ) == 1 ) {
            if ( 0 == strcmp( buffer, "</text>" ) ) break;
            if ( strlen(buffer) >= 4 ) currentDay->add( buffer );
        }
        availableDays.add( currentDay );

        for( int i=0; i < currentDay->getSize(); i++ ) {
            currentWords.push( currentDay->getWord(i) );
        }

        if ( availableDays.getSize() > 7 ) {

            oldestDay = availableDays.remove();

            for( int i=0; i < oldestDay->getSize(); i++ ) {
                currentWords.decreaseCounter( oldestDay->getWord(i) );
            }

            delete oldestDay;
        }
    }
    void showMostFrequentWords( int N )
    {
        Word *word;
        int counter=1<<30;

        Queue<Word> shownWords;

        fprintf( stdout, "<top %d>\n", N );

        while( --N >= 0 && currentWords.getSize() > 0 ) {

            word = currentWords.top(); currentWords.remove();

            counter = word->getCounter();

            fprintf( stdout, "%s %d\n", word->getWord(), counter );

            shownWords.add( word );
        }
        while( currentWords.getSize() > 0 ) {

            word = currentWords.top();

            if ( word->getCounter() < counter ) break;

            currentWords.remove();

            fprintf( stdout, "%s %d\n", word->getWord(), counter );

            shownWords.add( word );
        }

        fprintf( stdout, "</top>\n" );

        while( shownWords.getSize() > 0 ) {
            currentWords.pushAgain( shownWords.remove() );
        }
    }
};



int main( int argc, char *argv[] )
{
    TrendingTopic tt;

    tt.processInput( stdin );

    return EXIT_SUCCESS;
}
