
#if !defined( _UTILS_HASH_AND_LIST ) 
#define  _UTILS_HASH_AND_LIST 

template <class Type>
class HashAndListItem
{
private:
    Type            *datum;
    HashAndListItem *previous,
                    *next;
    int              posInHash;

public:
    HashAndListItem( Type * datum )
    {
        this->previous = NULL;
        this->datum = datum;
        this->next = NULL;
        this->posInHash = -1;
    }
    HashAndListItem( HashAndListItem *previous, Type * datum, HashAndListItem *next )
    {
        this->previous = previous;
        this->datum = datum;
        this->next = next;
        this->posInHash = -1;

        if ( previous != NULL ) previous->next = this;
        if ( next     != NULL ) next->previous = this;
    }
    ~HashAndListItem()
    {
        // DO NOTHING. The datum must be deallocated outside this class.
    }

    inline Type * get() { return datum; }

    inline HashAndListItem * getPrevious() { return previous; }
    inline HashAndListItem * getNext() { return next; }

    inline void set( Type * datum ) { this->datum = datum; }
    inline void setPrevious( HashAndListItem * ptr ) { this->previous = ptr; }
    inline void setNext( HashAndListItem * ptr ) { this->next = ptr; }

    inline void setPosInHash( int pos ) { this->posInHash = pos; }
    inline int  getPosInHash() { return this->posInHash; }
};

template <class Type>
class HashAndList
{
private:
    HashAndListItem<Type>   *first,
                            *last;
    HashAndListItem<Type>  **vector;
    int                      size, S, T;
    int                      colisionsCounter;

    void rehash()
    {
        delete [] vector;
        
        T = T << 1;
        vector = new HashAndListItem<Type> * [T];
        for( int i=0; i < T; i++ ) vector[i] = NULL;
        size=0;

        S = T/5;
        while( S < T && (T%S) == 0 ) S++;

        for( HashAndListItem<Type> *item = first; item != NULL; item = item->getNext() ) {
            insert( item );
        }
    }
    void clearList()
    {
        while( first != NULL ) {
            HashAndListItem<Type>   *temp = first;
            first = first->getNext();
            delete temp;
        }
        first = last = NULL;
    }
    void removeFromList( HashAndListItem<Type> * item )
    {
        HashAndListItem<Type> * previous = item->getPrevious();
        HashAndListItem<Type> * next     = item->getNext();

        if ( item == first ) {

            first = item->getNext();
            if ( first != NULL ) first->setPrevious( NULL );

        } else if ( item == last ) {

            last = item->getPrevious();
            if ( last != NULL ) last->setNext( NULL );

        } else {
            if ( previous != NULL ) previous->setNext( next );
            if (     next != NULL ) next->setPrevious( previous );
        }

        delete item;
    }

public:


    HashAndList( int initialSize=0 )
    {
        T = ( initialSize > 0 ) ? initialSize : (1 << 18);

        vector = new HashAndListItem<Type> * [T];
        first = last = NULL;

        S = T/5;
        while( S < T && (T%S) == 0 ) S++;

        clear( true );
    }
    ~HashAndList()
    {
        delete [] vector;

        clearList();
    }
    int getColisions() { return colisionsCounter; }
    int getSize() { return size; }

    void clear( bool doMemset )
    {
        size=0;
        colisionsCounter=0;

        if ( doMemset ) {
            for( int i=0; i < T; i++ ) vector[i] = NULL;
        }

        clearList();
    }


    void insert( Type * datum )
    {
        int pos = find( datum );

        if ( vector[pos] == NULL ) {

            HashAndListItem<Type> * item = new HashAndListItem<Type>( last, datum, NULL );
            last = item;
            if ( 0 == size ) { first = item; }

            vector[pos] = item;
            item->setPosInHash( pos );

            ++size;

        } else {
            // DO NOTHING IF THE datum ALREADY EXISTS
        }
    }
    int find( Type * datum )
    {
        int pos = datum->getHashCode() % T;

        while( vector[pos] != NULL && !datum->equals( vector[pos]->get() ) ) {

            pos = (pos+S)%T;
        }
        return pos;
    }

    Type * get( Type * datum )
    {
        int pos = find(datum);
        return (vector[pos] != NULL ) ? vector[pos]->get() : NULL;
    }

    void remove( int pos )
    {
        if ( size > 0 ) {
            HashAndListItem<Type> * item = vector[pos];
            vector[pos] = NULL;
            --size;

            removeFromList( item );
        }
    }


    class iterator {
        private:
            int index;
            HashAndListItem<Type> * currentItem;

        public:
            iterator( int pos=0, HashAndListItem<Type> *item=NULL ) : index(pos), currentItem(item) {}

            iterator( const iterator & iter ) : index(iter.index), currentItem(iter.currentItem) {}

            // Prefix operators
            iterator & operator++() { ++index; currentItem=currentItem->getNext();     return *this; }
            iterator & operator--()
            {
                --index;
                currentItem = (index == size) ? last : currentItem->getPrevious();
                return *this;
            }
/*
    Suffix operators
            iterator   operator++(int) { iterator temp(*this); ++(*this); return temp; }
            iterator   operator--(int) { iterator temp(*this); --(*this); return temp; }
*/
            bool operator!=( iterator  it ) { return index != it.index; }
            bool operator<=( iterator  it ) { return index <= it.index; }
            bool operator<(  iterator  it ) { return index <  it.index; }

            Type * operator*() { return currentItem->get(); }
    };

    iterator begin() { return iterator( 0, first ); }
    iterator end()   { return iterator( size, NULL ); }
};

#endif // _UTILS_HASH_AND_LIST 
