
import java.util.*;

class Place
{
    private String  name;
    private int     floor, x, y;

    private int     posInArray; // Used in the test case generator
    private Place   predecessor; // Used in the test case generator 
    private double  pathDistance; // Used in the test case generator 

    private boolean processed,
                    inQueue;

    private LinkedList<Place>   neighbours;

    public Place( String type, int floor, int x, int y )
    {
        name = String.format( "%s%02d%04d%04d", type, floor, x, y );

        this.floor = floor;
        this.x = x;
        this.y = y;

        this.processed=false;
        this.inQueue=false;

        this.neighbours = new LinkedList<Place>();
    }
    public Place( Scanner sf )
    {
        this.floor = sf.nextInt();
        this.x     = sf.nextInt();
        this.y     = sf.nextInt();
//        this.name  = sf.next();
        this.name  = String.format( "%03d%05d%05d", floor, x, y );

        this.processed=false;
        this.inQueue=false;

        this.neighbours = new LinkedList<Place>();
    }

    public String getName() { return name; }
    public int    getFloor() { return floor; }
    public int    getX() { return x; }
    public int    getY() { return y; }

    public int getPosInArray() { return posInArray; }
    public void setPosInArray( int pos ) { posInArray = pos; }
    public Place getPredecessor() { return predecessor; }
    public void setPredecessor( Place place ) { predecessor = place; }
    public double getPathDistance() { return pathDistance; }
    public void setPathDistance( double d ) { pathDistance = d; }

    public boolean getProcessed() { return processed; }
    public void    setProcessed( boolean p ) { processed=p; }
    public boolean getInQueue() { return inQueue; }
    public void    setInQueue( boolean q ) { inQueue=q; }

    public double distance( Place other )
    {
        int df = (this.floor - other.floor)*5;
        int dx = this.x - other.x;
        int dy = this.y - other.y;

        return Math.sqrt( df*df + dx*dx + dy*dy );
    }

    public void addNeighbour( Place other )
    {
        if ( other == this ) return;

        boolean exists=false;
        for( Place neighbour : neighbours ) {
            if ( neighbour == other ) {
                exists=true;
                break;
            }
        }

        if ( !exists ) {
            this.neighbours.add( other );
            other.neighbours.add( this );
        }
    }
    public int getCounterOfNeighbours()
    {
        return neighbours.size();
    }
    public LinkedList<Place> getNeighbours()
    {
        return neighbours;
    }

    public String toString()
    {
        //return String.format( "%d %d %d %s", floor, x, y, name );
        return String.format( "%d %d %d", floor, x, y );
    }
}


class Connection
{
    private static Random r = new Random();
    private static String types[] = { "walking", "stairs", "lift", "escalator" };

    private Place   from, to;
    private String  type;
    private double  distance;

    public Connection( Place from, Place to )
    {
        this.from = from;
        this.to   = to;
        if ( from.getFloor() == to.getFloor() ) {
            this.type = "walking";
        } else if ( from.getX() == to.getX()  &&  from.getY() == to.getY() ) {
            this.type = types[r.nextInt(2)];
        } else {
            this.type = types[2];
        }

        this.distance = from.distance( to );
    }
    public Connection( Place from, Place to, String type )
    {
        this.from = from;
        this.to   = to;
        this.type = type;
        this.distance = from.distance( to );
    }

    public double getDistance() { return distance; }
    public Place  getFrom() { return from; }
    public Place  getTo() { return to; }
    public String getType() { return type; }

    public String toString()
    {
        //return String.format( "%s %s %s", from.getName(), to.getName(), type );
        return String.format( "%d %d %s", from.getPosInArray(), to.getPosInArray(), type );
    }
}

public class JonSolutionA
{
    private static final double INFINITY = 9.9999e+300;
    private static final double EPSILON = 1.0e-15;

    private Place                       places[];
    private Hashtable<String,Integer>   nameToIndex; // Hash table to map place names with their index

    private double                      direct_distances[][];
    private double                      distances[][];
    private int                         bestPathTo[][];


    public JonSolutionA()
    {
        nameToIndex = new Hashtable<String,Integer>();        
        places = null;
    }

    public Place [] getPlaces()
    {
        return places;
    }
    public double [][] getDirectDistances()
    {
        return direct_distances;
    }
    public int getIndexOf( String placeName )
    {
        return nameToIndex.get( placeName );
    }

    public void loadFromScanner( Scanner sf )
    {
        int N = sf.nextInt();
        int M = sf.nextInt();

        places = new Place [N];
        direct_distances = new double [N][N];
        distances = new double [N][N];
        bestPathTo = new int [N][N];

        for( int i=0; i < N; i++ ) {
            direct_distances[i][i] = 0.0;
            bestPathTo[i][i] = i;
            for( int j=i+1; j < N; j++ ) {
                direct_distances[i][j] = direct_distances[j][i] = INFINITY;
                bestPathTo[i][j] = bestPathTo[j][i] = -1;
            }
        }

        for( int n=0; n < N; n++ ) {
            places[n] = new Place( sf );
            nameToIndex.put( places[n].getName(), n );
            places[n].setPosInArray( n );
        }

        for( int m=0; m < M; m++ ) {

//            String fromName = sf.next(); int from = nameToIndex.get( fromName );
//            String   toName = sf.next(); int   to = nameToIndex.get(   toName );
            int from = sf.nextInt();
            int to   = sf.nextInt();
            String typeName = sf.next();


            // Sanity check
            if ( direct_distances[from][to] != INFINITY ||  direct_distances[to][from] != INFINITY )
                // throw new RuntimeException( "Connection appears more than once: " + fromName + " --> " + toName + " (" + typeName + ")"  );
                throw new RuntimeException( "Connection appears more than once: " + from + " --> " + to + " (" + typeName + ")"  );

            places[from].addNeighbour( places[to] );
            places[to].addNeighbour( places[from] );

            direct_distances[from][to] = direct_distances[to][from] = places[from].distance( places[to] );

            if ( typeName.equals( "escalator" ) ) {
                direct_distances[from][to] = 1.0;
                direct_distances[to][from] *= 3.0;
            } else if ( typeName.equals( "lift" ) ) {
                direct_distances[from][to] = 1.0;
                direct_distances[to][from] = 1.0;
            }

            bestPathTo[from][to] = from;
            bestPathTo[to][from] = to;
        }
        for( int i=0; i < places.length; i++ ) {
            for( int j=0; j < places.length; j++ ) {
                distances[i][j] = direct_distances[i][j];
            }
        }
    }

    public void Floyd_Warshall_modified()
    {
        int changes;
        do {
            changes=0;

            for( int k=0; k < places.length; k++ ) {
                for( int i=0; i < places.length; i++ ) {
                    for( int j=0; j < places.length; j++ ) {

                        if ( i != j && i != k && j != k ) {

                            double through_k = distances[i][k] + direct_distances[k][j];
                            if ( through_k < distances[i][j] - EPSILON ) {
                                distances[i][j] = through_k;
                                bestPathTo[i][j] = k;
                                ++changes;
                            }
                        }
                    }
                }
            }

        } while( changes > 0 );
    }

    public void sanity_check()
    {
        int counter=0;
        for( int i=0; i < places.length; i++ ) {
            for( int j=0; j < places.length; j++ ) {
                if ( distances[i][j] == INFINITY || bestPathTo[i][j] == -1 ) {
                    System.err.println( "No existing path from " + places[i].getName() + " to " + places[j].getName() );
                    ++counter;
                }
            }
        }
        if ( counter > 0 ) throw new RuntimeException( "Several places are isolated!" );
    }

    public void bestPath( String fromName, String toName )
    {
        int from = nameToIndex.get( fromName );
        int   to = nameToIndex.get(   toName );

        //System.out.printf( " %10.4f ", distances[from][to] );

        bestPath( from, to );
        System.out.println();
    }
    public void bestPath( int from, int to )
    {
        if ( from != to ) {
            bestPath( from, bestPathTo[from][to] );
            System.out.print( " " + to );
        } else {
            System.out.print( from );
        }
    }

    public static void main( String args[] )
    {
        Scanner input = new Scanner( System.in );

        JonSolutionA solution = new JonSolutionA();

        solution.loadFromScanner( input );
        solution.Floyd_Warshall_modified();
        solution.sanity_check();

        int Q = input.nextInt();

        while( --Q >= 0  &&  input.hasNext() ) {
/*
            String from = input.next();
            String   to = input.next();
*/
            int from = input.nextInt();
            int   to = input.nextInt();

            solution.bestPath( from, to );
            System.out.println();
        }
    }
}
