
#include <vector>
#include <map>
#include <queue>
#include <stack>
#include <string>

#include <cassert>
#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <values.h>

const double MAX_DISTANCE = 9.99e+300;

using namespace std;

class Place {
private:
    int     f, x, y;
    string  name;
    int     posInArray;
    bool    inQueue;
    double  pathDistance;
    int     predecessor;

    vector<Place *> neighbours;

public:
    Place( int f, int x, int y, const char * name )
    {
        this->f = f;
        this->x = x;
        this->y = y;

        if ( name != NULL )
            this->name = name;

        this->pathDistance = 0.0;
    }
    Place( int f, int x, int y, string & name )
    {
        this->f = f;
        this->x = x;
        this->y = y;

        this->name = name;

        this->pathDistance = 0.0;
    }
    ~Place()
    {
        neighbours.clear();
    }

    inline int getFloor() { return f; }
    inline int getX() { return x; }
    inline int getY() { return y; }
    inline string & getName() { return name; }

    inline int getPosInArray() { return posInArray; }
    inline void setPosInArray( int pos ) { posInArray = pos; }

    inline bool isQueued() { return inQueue; }
    inline void setQueued( bool queued ) { inQueue = queued; }

    inline int getPredecessor() { return predecessor; }
    inline void setPredecessor( int p ) { predecessor=p; }

    inline double getPathDistance() { return pathDistance; }
    inline void setPathDistance( double d ) { pathDistance=d; }

    void addNeighbour( Place *p )
    {
        neighbours.push_back( p );
    }
    Place *getNeighbour( int i )
    {
        return neighbours[i];
    }
    unsigned int getNeighboursSize()
    {
        return neighbours.size();
    }

    double distance( Place *other )
    {
        double df = (this->f - other->f) * 5;
        double dx = (this->x - other->x);
        double dy = (this->y - other->y);

        return sqrt( df*df + dx*dx + dy*dy );
    }
};

class Solution
{
private:
    enum { WALKING=1, STAIRS, LIFT, ESCALATOR };

    map<string,int>     whichType;
    map<string,int>     nameToIndex;
    Place             **places;
    int                 N;

    double            **direct_distances;
    double            **distances;
    int               **bestPathTo;

    bool                floyd_computed;

public:

    Solution( FILE *input )
    {
        whichType["walking"]   = WALKING;
        whichType["stairs"]    = STAIRS;
        whichType["lift"]      = LIFT;
        whichType["escalator"] = ESCALATOR;

        int M;
        int rc;

        rc = fscanf( input, "%d%d", &N, &M );           assert( rc == 2 );

        places = new Place * [N];               memset( places, 0, N*sizeof(Place *) );

        direct_distances = new double * [N];
        direct_distances[0] = new double [N*N]; memset( direct_distances[0], 0, N*N*sizeof(double) );

        distances = new double * [N];
        distances[0] = new double [N*N];        memset( distances[0], 0, N*N*sizeof(double) );

        bestPathTo = new int * [N];
        bestPathTo[0] = new int [N*N];          memset( bestPathTo[0], 0, N*N*sizeof(int) );

        for( int i=1; i < N; i++ ) {
            direct_distances[i] = direct_distances[i-1] + N;
            distances[i] = distances[i-1] + N;
            bestPathTo[i] = bestPathTo[i-1] + N;
        }

        int  f, x, y;
        char buffer[256];

        for( int n=0; n < N; n++ ) {

            //rc = fscanf( input, "%d%d%d%s", &f, &x, &y, buffer );   assert( rc == 4 );
            //places[n] = new Place( f, x, y, buffer );
            //nameToIndex[buffer] = n;
            rc = fscanf( input, "%d%d%d", &f, &x, &y );   assert( rc == 3 );
            places[n] = new Place( f, x, y, NULL );
            places[n]->setPosInArray(n);
        }

        for( int i=0; i < N; i++ ) {
            direct_distances[i][i] = 0.0;
            distances[i][i] = 0.0;
            bestPathTo[i][i] = i;
            for( int j=i+1; j < N; j++ ) {
                direct_distances[i][j] = direct_distances[j][i] = MAX_DISTANCE;
                distances[i][j] = distances[j][i] = MAX_DISTANCE;
                bestPathTo[i][j] = bestPathTo[j][i] = -1;
            }
        }

        for( int m=0; m < M; m++ ) {
            
            // rc = fscanf( input, "%s", buffer );     assert( rc == 1 );
            // int from = nameToIndex[buffer];
            int from; rc = fscanf( input, "%d", &from );     assert( rc == 1 );

            // rc = fscanf( input, "%s", buffer );     assert( rc == 1 );
            // int   to = nameToIndex[buffer];
            int to; rc = fscanf( input, "%d", &to );     assert( rc == 1 );

            rc = fscanf( input, "%s", buffer );     assert( rc == 1 );
            int type = whichType[buffer];

            direct_distances[from][to] = direct_distances[to][from] = places[from]->distance( places[to] );
            bestPathTo[from][to] = from;
            bestPathTo[to][from] = to;

            if ( ESCALATOR == type ) {
                direct_distances[from][to] = 1.0;
                direct_distances[to][from] *= 3.0;
            } else if ( LIFT == type ) {
                direct_distances[from][to] = 1.0;
                direct_distances[to][from] = 1.0;
            }
        }

        for( int i=0; i < N; i++ ) {
            for( int j=0; j < N; j++ ) {
                distances[i][j] = direct_distances[i][j];
            }
        }

        floyd_computed = false;
    }
    ~Solution()
    {
        for( int i=0; i < N; i++ ) delete places[i];

        delete [] places;

        delete [] direct_distances[0];
        delete [] direct_distances;
        delete [] distances[0];
        delete [] distances;
        delete [] bestPathTo[0];
        delete [] bestPathTo;
    }

    void Floyd_Warshall_modified()
    {
        const double EPSILON = 1.0e-5;

        int changes;
        do {
            changes=0;

            //for( int k=0; k < N; ++k )
            for( int k=N-1; k >= 0; --k ) // For checking different compatible solutions
            {
                for( int i=0; i < N; i++ ) {
                    for( int j=0; j < N; j++ ) {

                        if ( i != j && i != k && j != k ) {

                            double through_k = distances[i][k] + direct_distances[k][j];
                            if ( through_k < distances[i][j] - EPSILON ) {
                                distances[i][j] = through_k;
                                bestPathTo[i][j] = k;
                                ++changes;
                            }
                        }
                    }
                }
            }
        } while( changes > 0 );

        floyd_computed=true;
    }
    void sanity_check()
    {
        for( int i=0; i < N; i++ ) {
            for( int j=0; j < N; j++ ) {
                if ( distances[i][j] == MAX_DISTANCE || bestPathTo[i][j] == -1 ) {
                    fprintf( stderr, "No existing path from %s to %s\n", places[i]->getName().c_str(), places[j]->getName().c_str() );
                    abort();
                }
            }
        }
    }

    void bestPath( int from, int to )
    {
        if ( from != to ) {
            bestPath( from, bestPathTo[from][to] );
            printf( " %s", places[to]->getName().c_str() );            
        } else {
            printf( "%s", places[from]->getName().c_str() );            
        }
    }

    void process_queries( FILE *input )
    {
        for( int i=0; i < N; i++ ) {
            for( int j=0; j < N; j++ ) {
                if ( i != j  &&  direct_distances[i][j] < MAX_DISTANCE ) {
                    places[i]->addNeighbour( places[j] );
                }
            }
        }
/*
        char buffer1[256], buffer2[256];

        while( fscanf( input, "%s%s", buffer1, buffer2 ) == 2 ) {
            
            int from = nameToIndex[buffer1];
            int   to = nameToIndex[buffer2];

            if ( floyd_computed ) {
                //printf( " %10.4f ", distances[from][to] );
                bestPath( from, to ); printf( "\n" );
            } else {
                Dijkstra( from, to );
            }
        }
*/
        int Q;
        int rc = fscanf( input, "%d", &Q ); assert( rc == 1 );

        int from, to;
        while( --Q >= 0  &&  fscanf( input, "%d%d", &from, &to ) == 2 ) {

            if ( floyd_computed ) {
                //printf( " %10.4f ", distances[from][to] );
                bestPath( from, to ); printf( "\n" );
            } else {
                Dijkstra( from, to );
            }
        }

    }
    void Dijkstra( int from, int to )
    {
        for( int i=0; i < N; i++ ) {
            places[i]->setQueued( false );
            places[i]->setPathDistance( MAX_DISTANCE );
            places[i]->setPredecessor( -1 );
        }

        queue<int>  alive;

        alive.push( from );
        places[from]->setQueued( true );
        places[from]->setPathDistance( 0.0 );
        places[from]->setPredecessor( -1 );

        int current = -1;

        while( alive.size() > 0 ) {

            current = alive.front(); alive.pop();
            places[current]->setQueued( false );

            //for( int i = (int)places[current]->getNeighboursSize()-1; i >= 0; --i )
            for( unsigned int i = 0; i < places[current]->getNeighboursSize(); ++i )
            {
                
                int next = places[current]->getNeighbour(i)->getPosInArray();

                double dist = places[current]->getPathDistance() + direct_distances[current][next];
                
                if ( dist < places[next]->getPathDistance()  &&  dist < places[to]->getPathDistance() ) {

                    places[next]->setPathDistance( dist );
                    places[next]->setPredecessor( current );
                    if ( ! places[next]->isQueued() ) {
                        alive.push( next );
                        places[next]->setQueued( true );
                    }
                }
            }
        }

        stack<int>  s;
        current=to;
        while( -1 != current ) {
            s.push( current );
            current = places[current]->getPredecessor();
        }

        current = s.top(); s.pop();
        // printf( "%s", places[current]->getName().c_str() );
        printf( "%d", current );
        while( ! s.empty() ) {
            current = s.top(); s.pop();
            //printf( " %s", places[current]->getName().c_str() );
            printf( " %d", current );
        }
        printf( "\n" );
    }

    int checkOutputCandidateWithPlaceNames( FILE *input, FILE *candidate )
    {
        char buffer1[256], buffer2[256];

        while( fscanf( input, "%s%s", buffer1, buffer2 ) == 2 ) {

            int from = nameToIndex[buffer1];
            int   to = nameToIndex[buffer2];

            int rc = fscanf( candidate, "%s", buffer1 ); if ( rc != 1 ) return -1;
            double distance=0.0;
            int current = nameToIndex[buffer1];
            while( current != to && fscanf( candidate, "%s", buffer1 ) == 1 ) {
                int next = nameToIndex[buffer1];
                distance += distances[current][next]; 
                current = next;
            }

            if ( fabs( distance - distances[from][to] ) > 1.0e-3 ) return -1;
        }

        return 0;
    }
    int checkOutputCandidate( FILE *input, FILE *candidate )
    {
        int from, to, current, next;

        while( fscanf( input, "%d%d", &from, &to ) == 2 ) {

            int rc = fscanf( candidate, "%d", &current ); if ( rc != 1 ) return -1;
            double distance=0.0;
            while( current != to && fscanf( candidate, "%d", &next ) == 1 ) {
                distance += direct_distances[current][next];  // WARNING: This is different from above
                current = next;
            }

            if ( fabs( distance - distances[from][to] ) > 1.0e-3 ) return -1;
        }

        return 0;
    }

};


int main( int argc, char *argv[] )
{
    Solution *sol = NULL;
    int rc = EXIT_SUCCESS;

    if ( argc == 1 ) { // Normal procedure

        sol = new Solution(stdin);
        /*
        sol->Floyd_Warshall_modified();
        sol->sanity_check();
        */
        sol->process_queries(stdin);

    } else if ( argc == 3 ) { // Act as checker

        FILE *fpWithInput  = fopen( argv[1], "r" );
        FILE *fpWithOutput = fopen( argv[2], "r" );

        sol = new Solution( fpWithInput );
        sol->Floyd_Warshall_modified();
        sol->sanity_check();

        int rc = sol->checkOutputCandidate( fpWithInput, fpWithOutput );

        fclose( fpWithInput );
        fclose( fpWithOutput );

        if ( rc == 0 )
            fprintf( stdout, "OK\n" );
        else {
            fprintf( stdout, "DIFFERENT\n" );
            rc = EXIT_FAILURE;
        }

    } else {
        return EXIT_FAILURE;
    }

    delete sol;

    return rc;
}
