
#include <cstdio>
#include <cstdlib>
#include <cmath>
#include <cstring>
#include <cctype>
#include <cassert>

#include <list>
#include <vector>


inline int max( int a, int b ) { return a > b ? a : b; }
inline int min( int a, int b ) { return a < b ? a : b; }

class HashNode
{
public:
    char    *key;
    void    *datum;

    HashNode( char *key, void *datum )
    {
        this->key = key;
        this->datum = datum;
    }
};

class HashMap
{
private:
    double loadFactor;
    int    N, n;

    HashNode  **A;

    // hash function extracted from:
    // http://stackoverflow.com/questions/114085/fast-string-hashing-algorithm-with-low-collision-rates-with-32-bit-integer
    int hashCode( const char * s )
    {
        unsigned int hash = 0;
        for(; *s; ++s) {
            hash += *s;
            hash += (hash << 10);
            hash ^= (hash >> 6);
        }
        hash += (hash << 3);
        hash ^= (hash >> 11);
        hash += (hash << 15);
        return hash%N;
    }

    void rehash()
    {
        HashNode **aux = A;

        int i = N-1;
        N = 3*N+1;
        A = new HashNode * [N];
        memset( A, 0, N * sizeof( HashNode * ) );

        for( ; i >= 0; --i ) {
            
            if ( aux[i] != NULL ) {

                int pos = hashCode( aux[i]->key );

                while( A[pos] != NULL  &&  strcmp( A[pos]->key, aux[i]->key )!=0 )
                    pos = (pos+1)%N;

                assert( A[pos] == NULL );
                A[pos] = aux[i];
            }
        }

        delete [] aux;
    }

public:

    HashMap( int N=89, double loadFactor=0.75 )
    {
        this->n = 0;
        this->N = N;
        this->loadFactor = loadFactor;
        this->A = new HashNode * [N];

        memset( this->A, 0, N * sizeof( HashNode * ) );
    }
    ~HashMap()
    {
        for( int pos=0; pos < N; pos++ )
            if ( A[pos] != NULL ) delete A[pos];

        delete [] A;
    }

    void put( char *key, void *obj )
    {
        if ( (double)n/N >= loadFactor ) rehash();

        int pos = hashCode(key);

        while( A[pos] != NULL  &&  strcmp( A[pos]->key, key )!=0 )
            pos = (pos+1)%N;

        if ( A[pos] != NULL ) A[pos] = new HashNode( key, obj );

        ++n;
    }
    void *get( char *key )
    {
        int pos = hashCode(key);

        while( A[pos] != NULL  &&  strcmp( A[pos]->key, key )!=0 )
            pos = (pos+1)%N;

        return ( A[pos] == NULL ) ? NULL : A[pos]->datum;
    }

    void checkAndPut( char *key, void *obj )
    {
        if ( (double)n/N >= loadFactor ) rehash();

        int pos = hashCode(key);

        while( A[pos] != NULL  &&  strcmp( A[pos]->key, key )!=0 )
            pos = (pos+1)%N;

        if ( A[pos] == NULL ) A[pos] = new HashNode( key, obj );

        ++n;
    }
};

// Definition of class Trie and related classes for efficient storing of words
class TrieNode
{
public:
    static const int alphabetSize = 'z' - 'a' + 1;

    TrieNode **nextNodes;
    char      *word;
    void      *aux;

public:
    static char * duplicate( char *s )
    {
        char *t = new char [ strlen( s ) + 1 ];

        strcpy( t, s );

        return t;
    }


    TrieNode( char *word = NULL )
    {
        nextNodes = new TrieNode * [alphabetSize];

        for( int i=0; i < alphabetSize; i++ ) nextNodes[i] = NULL;

        this->word = NULL;
        setWord( word );

        aux = NULL;
    }
    ~TrieNode()
    {
        for( int i=0; i < alphabetSize; i++ )
            if ( nextNodes[i] != NULL ) delete nextNodes[i];

        delete [] nextNodes;
        resetWord();
    }

    void resetWord()
    {
        if ( word != NULL ) delete [] word;
        word = NULL;
    }
    void setWord( char *word )
    {
        resetWord();
        this->word = ( word == NULL ) ? NULL : duplicate(word);
    }

    inline void addNext( TrieNode *nextNode, char symbol )
    {
        this->nextNodes[symbol-'a'] = nextNode;
    }

    inline TrieNode *getNext( char symbol )
    {
        return this->nextNodes[symbol-'a'];
    }

    inline char *getWord() { return word; }

    inline bool isTerminal() { return (word != NULL); }

    inline void setAux( void *aux ) { this->aux = aux; }
    inline void *getAux() { return this->aux; }
};
class Trie
{
private:
    TrieNode    *root;

public:
    Trie()
    {
        root = new TrieNode();
    }
    ~Trie()
    {
        delete root;
    }

    TrieNode *findOrCreatePath( char *word, bool toCreate )
    {
//        assert( word != NULL );
//        assert( strlen(word) > 0 );

        TrieNode *node = root,
                 *nextNode = NULL;

        for( int i=0; word[i] != '\0' && node != NULL; i++ ) {
            
            nextNode = node->getNext( word[i] );

            if ( nextNode == NULL && toCreate ) {
                nextNode = new TrieNode();
                node->addNext( nextNode, word[i] );
            }

            node = nextNode;
        }

        return node;
    }
    bool checkAndAdd( char *word, void *aux=NULL )
    {
        TrieNode *node = findOrCreatePath( word, true );

        if ( node->isTerminal() ) return true;

        node->setWord( word );
        node->setAux( aux );

        return false;
    }

    void *checkAndGetAux( char *word )
    {
        TrieNode *node = findOrCreatePath( word, false );

        return (node != NULL ) ? node->getAux() : NULL;
    }
};


// Definition of class NGram and related classes for storing efficiently sequences of words
class NGramNode
{
#define __USE_HASH__ 0
private:
#if !defined( __USE_HASH__ )
    Trie *nextLevelWords;
    Trie *nextWords;
#else
    HashMap nextLevelWords;
    HashMap nextWords;
#endif
    char *word;
    int   level;

public:

    NGramNode( char *word, int level )
    {
#if !defined( __USE_HASH__ )
        nextLevelWords = new Trie();
             nextWords = new Trie();
#endif

        this->word = (word != NULL) ? TrieNode::duplicate( word ) : NULL;
        this->level = level;
    }
    ~NGramNode()
    {
        if ( word != NULL ) delete [] word;

#if !defined( __USE_HASH__ )
        delete nextLevelWords;
        delete nextWords;
#endif
    }

    inline int getLevel() { return level; }
    inline char *getWord() { return word; }

    inline void setNext( NGramNode * next )
    {
#if !defined( __USE_HASH__ )
        nextWords->checkAndAdd( next->word, (void *)next );
#else
        nextWords.checkAndPut( next->word, (void *)next );
#endif
    }
    inline void setNextLevel( NGramNode * next )
    {
#if !defined( __USE_HASH__ )
        nextLevelWords->checkAndAdd( next->word, (void *)next );
#else
        nextLevelWords.checkAndPut( next->word, (void *)next );
#endif
    }

    inline NGramNode *getNextLevel( char *word )
    {
#if !defined( __USE_HASH__ )
        return (NGramNode *) nextLevelWords->checkAndGetAux( word );
#else
        return (NGramNode *) nextLevelWords.get( word );
#endif
    }
    inline NGramNode *getNextInTheSameLevel( char *word )
    {
#if !defined( __USE_HASH__ )
        return (NGramNode *) nextWords->checkAndGetAux( word );
#else
        return (NGramNode *) nextWords.get( word );
#endif
    }
};

class NGram
{
private:
    NGramNode *root,
              *current;
    
    std::list<NGramNode *> listOfNodesForDeletion;

public:

    NGram()
    {
        init();
    }
    ~NGram()
    {
        clear();
    }

    void clear()
    {
        while( !listOfNodesForDeletion.empty() ) {
            delete listOfNodesForDeletion.front();
            listOfNodesForDeletion.pop_front();
        }
    }
    void init()
    {
        root = new NGramNode( NULL, 0 );
        current = NULL;
        listOfNodesForDeletion.push_back( root );
    }

    void reset() { clear(); init(); }

    void startSearch()
    {
        current = root;
    }

    bool checkAndAddWord( char *word ) 
    {
        NGramNode *next = root->getNextLevel( word );

        if ( next != NULL ) return true;

        next = new NGramNode( word, root->getLevel()+1 );
        root->setNextLevel( next );

        listOfNodesForDeletion.push_back( next );

        return false;
    }

    NGramNode * findOrCreateNGram( std::vector<char *> & sentence, unsigned int first, unsigned int last, int topLevel )
    {
        NGramNode *current = root;

        for( unsigned int i=first; i <= last; i++ ) {

            char *word = sentence[i];

            NGramNode *next = ( current->getLevel() < topLevel )
                            ? current->getNextLevel( word )
                            : current->getNextInTheSameLevel( word );

            if ( next == NULL ) {
                if ( current->getLevel() < topLevel ) {
                    next = new NGramNode( word, current->getLevel()+1 );
                    current->setNextLevel( next );

                    listOfNodesForDeletion.push_back( next );
                } else {
                    next = findOrCreateNGram( sentence, first+1, i, topLevel );
                    current->setNext( next );
                }
            }

            current = next;
        }

        return current;
    }

    bool checkSentence( std::vector<char *> & sentence, unsigned int first, unsigned int last, const int topLevel )
    {
        NGramNode *current = root;

        for( unsigned int i=first; i <= last && current != NULL; i++ ) {

            current = ( current->getLevel() < topLevel )
                    ? current->getNextLevel( sentence[i] )
                    : current->getNextInTheSameLevel( sentence[i] );
        }

        return current != NULL;
    }
};


static char buffer[4096];

static bool isPunctuation( int ch )
{
    return (ch == '.' || ch == ',' || ch == ';' || ch == ':' || ch == '?' || ch == '!');
}
static char *nextToken()
{
    int ch, i=0;

    while( (ch=getchar()) != EOF && isspace(ch) ) ;

    if ( ch == EOF ) return NULL;

    buffer[i++] = ch;
    while( (ch=getchar()) != EOF && !isspace(ch) ) {
        buffer[i++] = ch;
        if ( isPunctuation(ch) ) break;
    }
    buffer[i++] = '\0';

    return buffer;
}

char *extractPlainWord( char *word )
{
    int l = strlen( word );

    char *plainWord = new char [l+1];

    int j=0;

    for( int i=0; i < l; i++ ) {
        if ( !isPunctuation( word[i] ) ) {
            plainWord[j++] = word[i];
        }
    }
    plainWord[j] = '\0';

    if ( j == 0 ) {
        delete [] plainWord;
        plainWord = NULL;
    }
    return plainWord;
}
char *toLowerCase( char *word )
{
    int l = strlen( word );

    char *wordLowerCase = new char [l+1];

    for( int i=0; i < l; i++ ) wordLowerCase[i] = tolower( word[i] );
    wordLowerCase[l] = '\0';

    return wordLowerCase;
}

typedef enum { OUT, PREVIOUS_KNOWLEDGE, CONVERSATION } States;

int main( int argc, char *argv[] )
{
    NGram    language;
    char    *word;
    int      complexityLevel = 0;
    int      testNumber = 1;
    bool     unknownSentence = false;

    States   state=OUT;

    std::list<char *>    sentence;
    std::vector<char *>  plainSentence;


    while( (word=nextToken()) != NULL ) {

        switch( word[0] ) {

            case '*' : 
                assert( state == PREVIOUS_KNOWLEDGE );
                state = CONVERSATION;
                language.startSearch();
                break;

            case '#' :
                assert( state == CONVERSATION );
                state = OUT;
                break;

            default :
                if ( state == OUT ) {

                    complexityLevel = atoi( word );     assert( 2 <= complexityLevel && complexityLevel <= 5 );
                    state = PREVIOUS_KNOWLEDGE;
                    if ( testNumber > 1 ) printf( "\n" );
                    printf( "Learning case %d\n", testNumber++ );
                    language.reset();
                    language.checkAndAddWord( (char*)"joe" );

                } else {

                    char *plainWord = extractPlainWord( word );

                    if ( plainWord != NULL ) {

                        char *plainWordLowerCase = toLowerCase( plainWord );

                        sentence.push_back( plainWord );
                        plainSentence.push_back( plainWordLowerCase );

                        if ( language.checkAndAddWord( plainWordLowerCase ) == false ) {
                            if ( state == CONVERSATION ) printf( "What does the word \"%s\" mean?\n", plainWord );
                        }
                    }

                    if ( isPunctuation( word[strlen(word)-1] ) ) {

                        unsigned int last = plainSentence.size();

                        if ( last > 1 ) {

                            if ( language.checkSentence( plainSentence, 0, last-1, complexityLevel-1 ) == false ) {

                                if ( state == CONVERSATION ) {
                                    printf( "What does the sentence \"" );
                                    std::list<char *>::iterator iter=sentence.begin();
                                    printf( "%s", *iter++ );
                                    for( ; iter != sentence.end(); iter++ ) printf( " %s", *iter );
                                    printf( "\" mean?\n" );
                                }

                                language.findOrCreateNGram( plainSentence, 0, last-1, complexityLevel-1 );
                            }
                        }

                        while( !sentence.empty() ) {
                            delete [] sentence.front();
                            sentence.pop_front();
                        }

                        for( unsigned int i=0; i < plainSentence.size(); i++ ) delete [] plainSentence[i];
                        plainSentence.clear();

                        unknownSentence = false;
                        language.startSearch();
                    }
                } // ELSE del IF ( state == OUT ) 
        } // SWITCH( word[0] )
    } // WHILE
    

    return 0;
}
