
import java.util.*;

class TrieNode
{
    public static final int alphabetSize = 'z' - 'a' + 1;

    private TrieNode  nextNodes[];
    private String    word;
    private Object    aux;

    public TrieNode( String word )
    {
        this.word = word;

        nextNodes = new TrieNode [alphabetSize];

        aux = null;
    }

    public void setWord( String word )
    {
        this.word = word;
    }

    public void addNext( TrieNode nextNode, char symbol )
    {
        this.nextNodes[symbol-'a'] = nextNode;
    }

    public TrieNode getNext( char symbol )
    {
        return this.nextNodes[symbol-'a'];
    }

    public String getWord() { return word; }
    public boolean isTerminal() { return (word != null); }

    public void setAux( Object aux ) { this.aux = aux; }
    public Object getAux() { return this.aux; }
}

class Trie
{
    private TrieNode  root;

    public Trie()
    {
        root = new TrieNode( null );
    }

    public TrieNode findOrCreatePath( String word, boolean toCreate )
    {
        TrieNode     node = root,
                 nextNode = null;

        int l = word.length();

        for( int i=0; i < l && node != null; i++ ) {
            
            nextNode = node.getNext( word.charAt(i) );

            if ( nextNode == null && toCreate ) {
                nextNode = new TrieNode( null );
                node.addNext( nextNode, word.charAt(i) );
            }

            node = nextNode;
        }

        return node;
    }

    public boolean checkAndAdd( String word, Object aux )
    {
        TrieNode node = findOrCreatePath( word, true );

        if ( node.isTerminal() ) return true;

        node.setWord( word );
        node.setAux( aux );

        return false;
    }

    public Object checkAndGetAux( String word )
    {
        TrieNode node = findOrCreatePath( word, false );

        return (node != null ) ? node.getAux() : null;
    }
}

interface NGramNode
{
    public int getLevel();
    public String getWord();

    public void setNext( NGramNode next );
    public void setNextLevel( NGramNode next );
    public NGramNode getNextLevel( String word );
    public NGramNode getNextInTheSameLevel( String word );
}

class NGramNodeWithTrie
    implements NGramNode
{
    private Trie    nextLevelWords;
    private Trie    nextWords;
    private String  word;
    private int     level;

    public NGramNodeWithTrie( String word, int level )
    {
        nextLevelWords = new Trie();
             nextWords = new Trie();

        this.word  = word;
        this.level = level;
    }

    public int getLevel() { return level; }
    public String getWord() { return word; }

    public void setNext( NGramNode next )
    {
        nextWords.checkAndAdd( next.getWord(), (Object)next );
    }
    public void setNextLevel( NGramNode next )
    {
        nextLevelWords.checkAndAdd( next.getWord(), (Object)next );
    }

    public NGramNode getNextLevel( String word )
    {
        return (NGramNode)nextLevelWords.checkAndGetAux( word );
    }
    public NGramNode getNextInTheSameLevel( String word )
    {
        return (NGramNode)nextWords.checkAndGetAux( word );
    }
}

class NGramNodeWithHash
    implements NGramNode
{
    private HashMap<String,NGramNode> nextLevelWords;
    private HashMap<String,NGramNode> nextWords;
    private String  word;
    private int     level;

    public NGramNodeWithHash( String word, int level )
    {
        nextLevelWords = new HashMap<String,NGramNode>();
             nextWords = new HashMap<String,NGramNode>();

        this.word  = word;
        this.level = level;
    }

    public int getLevel() { return level; }
    public String getWord() { return word; }

    public void setNext( NGramNode next )
    {
        if ( !nextWords.containsKey( next.getWord() ) )
            nextWords.put( next.getWord(), next );
    }
    public void setNextLevel( NGramNode next )
    {
        if ( !nextLevelWords.containsKey( next.getWord() ) )
            nextLevelWords.put( next.getWord(), next );
    }

    public NGramNode getNextLevel( String word )
    {
        return nextLevelWords.get( word );
    }
    public NGramNode getNextInTheSameLevel( String word )
    {
        return nextWords.get( word );
    }
}

class NGram
{
    private NGramNode root, current;
    
    public NGram()
    {
        init();
    }
    private void init()
    {
        //root = new NGramNodeWithTrie( null, 0 );
        root = new NGramNodeWithHash( null, 0 );
        current = null;
    }

    public void reset() { init(); }

    public boolean checkAndAddWord( String word ) 
    {
        NGramNode next = root.getNextLevel( word );

        if ( next != null ) return true;

        if ( root instanceof NGramNodeWithTrie )
            next = new NGramNodeWithTrie( word, root.getLevel()+1 );
        else
            next = new NGramNodeWithHash( word, root.getLevel()+1 );

        root.setNextLevel( next );

        return false;
    }

    public NGramNode findOrCreateNGram( ArrayList<String> sentence, int first, int last, int topLevel )
    {
        NGramNode current = root;

        if ( first > last ) {
            System.err.printf( "findOrCreateNGram( *, %d, %d, %d )\n", first, last, topLevel );
            System.exit( -1 );
        }

        for( int i=first; i <= last; i++ ) {

            String word = sentence.get(i);

            NGramNode next = ( current.getLevel() < topLevel )
                           ? current.getNextLevel( word )
                           : current.getNextInTheSameLevel( word );

            if ( next == null ) {

                if ( current.getLevel() < topLevel ) {

                    if ( root instanceof NGramNodeWithTrie )
                        next = new NGramNodeWithTrie( word, current.getLevel()+1 );
                    else
                        next = new NGramNodeWithHash( word, current.getLevel()+1 );

                    current.setNextLevel( next );

                } else {

                    next = findOrCreateNGram( sentence, first+1, i, topLevel );
                    current.setNext( next );
                }
            }

            current = next;
        }

        if ( current == root  ) {
            System.err.printf( "findOrCreateNGram( *, %d, %d, %d ) --> root \n", first, last, topLevel );
            System.exit( -1 );
        }
        if ( current.getLevel() != Math.min( (last-first+1), topLevel ) ) {
            System.err.printf( "findOrCreateNGram( *, %d, %d, %d ) != current.getLevel()=%d vs topLevel=%d\n", first, last, topLevel, current.getLevel(), topLevel );
            System.exit( -1 );
        }

        return current;
    }

    public boolean checkSentence( ArrayList<String> sentence, int first, int last, int topLevel )
    {
        NGramNode current = root;

        for( int i=first; i <= last && current != null; i++ ) {

            current = ( current.getLevel() < topLevel )
                    ? current.getNextLevel( sentence.get(i) )
                    : current.getNextInTheSameLevel( sentence.get(i) );
        }

        return current != null;
    }
}

class AdvancedSolution
{
    enum States { OUT, READING_PREVIOUS_KNOWLEDGE, PROCESSING_CONVERSATION };

    public static void main( String args[] )
    {
        Map<String,Integer> words = new HashMap<String,Integer>();
        NGram               language = new NGram();
        String              buffer,
                            of, learnt, sentences,
                            knowledge;
        int                 complexityLevel=2,
                            caseNumber=1;

        LinkedList<String>  listOfPendingWords=null;
        ArrayList<String>   sentence = new ArrayList<String>();
        ArrayList<String>   plainSentence = new ArrayList<String>();

        States              state;
        boolean             unknownSentence=false;
        char                lastChar;
        
        Scanner sf = new Scanner( System.in );
        
        state = States.OUT;

        while( sf.hasNext() ) {
            
            buffer = sf.next();

            switch( state ) {
                case OUT :
                    complexityLevel = Integer.parseInt( buffer );
                    words = new HashMap<String,Integer>();
                    words.put( "joe", 1 );
                    language = new NGram();
                    state = States.READING_PREVIOUS_KNOWLEDGE;
                    sentence.clear();
                    plainSentence.clear();
                    if ( caseNumber > 1 ) System.out.println();
                    System.out.println( "Learning case " + caseNumber );
                    ++caseNumber;
                    break;

                case READING_PREVIOUS_KNOWLEDGE :
                case PROCESSING_CONVERSATION :

                    lastChar = buffer.charAt( buffer.length() - 1 );

                    if ( lastChar != '*' && lastChar != '#' ) {

                        String cleanCaseSensitiveWord = removeNonAlphaCharacters( buffer );

                        if ( cleanCaseSensitiveWord.length() > 0 ) {
                            String cleanWord = cleanCaseSensitiveWord.toLowerCase();

                            if ( !words.containsKey( cleanWord ) ) {
                                if ( state == States.PROCESSING_CONVERSATION )
                                    System.out.println( "What does the word \"" + cleanCaseSensitiveWord + "\" mean?" );

                                words.put( cleanWord, words.size()+1 );
                            }

                            sentence.add( cleanCaseSensitiveWord );
                            plainSentence.add( cleanWord );

                        }
                    }

                    if ( lastChar == '.' || lastChar == ':'
                      || lastChar == ',' || lastChar == ';' 
                      || lastChar == '?' || lastChar == '!'
                      || lastChar == '#' || lastChar == '*' ) {


                        int last = plainSentence.size();
                        if ( last > 1 && language.checkSentence( plainSentence, 0, last-1, complexityLevel-1 ) == false ) {

                            if ( state == States.PROCESSING_CONVERSATION  &&  sentence.size() > 0 ) {

                                String key = sentence.get(0);
                                for( int i=1 ; i < sentence.size(); ++i ) key += " " + sentence.get(i);

                                System.out.println( "What does the sentence \"" + key + "\" mean?" );
                            }

                            language.findOrCreateNGram( plainSentence, 0, last-1, complexityLevel-1 );
                        }
                        sentence.clear();
                        plainSentence.clear();
                        unknownSentence = false;
                    }

                    if ( lastChar == '#' ) state = States.OUT;
                    if ( lastChar == '*' ) state = States.PROCESSING_CONVERSATION;

                    break;
            } // switch( state )
        } // while()
    }

    private static String removeNonAlphaCharacters( String in )
    {
        String out = "";
        
        for( int i=0; i < in.length(); i++ ) {
            char ch = in.charAt(i);
            if ( Character.isLetter( ch ) || Character.isDigit( ch ) ) {
                out += ch;
            }
        }

        return out;
    }
}
