// Checker to be used by evaluator.
//
// Usage: [checker] [input] [official_output] [contestant_output]
//
// Score (real between 0.0 and 1.0) written on stdout.
// Textual description of the result written on stderr.

#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <set>

using namespace std;

/**
 * @param p fraction of points awarded to the contestant.
 * @pararm m error message displayed to the contestant.
 */
void finish(double p, const string& m);

/**
 * The main checking function.
 * @param fin official input
 * @param foff official output
 * @param fout contestant's output
 */
void checker(ifstream& fin, ifstream& foff, ifstream& fout)
{
  const string WRONG_OUTPUT_FORMAT = "Incorrectly formatted output.";
  const string TEST_DATA_ERROR = "Error in official input or output.";
  const string WRONG = "Final key weight is smaller than 2n.";
  const string TOO_MANY_OPERATIONS = "Too many operations.";
  const string INVALID_OPERATION = "Invalid operation.";
  const string CORRECT = "Correct.";

  // Read official input
  string S;

  if (!(fin >> S)) finish(0, TEST_DATA_ERROR);

  int triN = S.size();
  if (triN % 3) finish(0, TEST_DATA_ERROR);
  int N = triN / 3;

  // Convert input to numbers 0/1
  for (int i = 0; i < 3*N; ++i) {
    S[i] -= '0';
  }
  
  // Read contestant's output
  int M;
  if (!(fout >> M)) finish(0, WRONG_OUTPUT_FORMAT);

  if (M < 0) finish(0, WRONG_OUTPUT_FORMAT);
  if (M > N) finish(0, TOO_MANY_OPERATIONS);

  for (int i = 0; i < M; ++i) {
    int x;
    if (!(fout >> x)) finish(0, WRONG_OUTPUT_FORMAT);
    if (!(1 <= x && x <= 3*N-1)) finish(0, INVALID_OPERATION);
    --x;
    S[x] ^= 1;
    S[x+1] ^= 1;
  }
  
  string tmp_buf;
  if (fout >> tmp_buf) finish(0, WRONG_OUTPUT_FORMAT);

  int cnt = 1;
  for (int i = 0; i < 3*N-1; ++i) {
    cnt += S[i] ^ S[i+1];
  }

  if (cnt < 2*N) finish(0, WRONG);

  finish(1, CORRECT);
  // The function MUST terminate before this line via finish()!
}

void finish(double p, const string& m) {
  cout << p << endl;
  cout << m << endl;
  exit(0);
}

int main(int argc, char *argv[])
{
  assert(argc == 4);

  ifstream fin(argv[1]);
  ifstream foff(argv[2]);
  ifstream fout(argv[3]);

  assert(!fin.fail() && !fout.fail());
  checker(fin, foff, fout);
  assert(false); // checker must terminate via finish() before exiting!

  return 0;
}
