// Checker to be used by evaluator.
//
// Usage: [checker] [input] [official_output] [contestant_output]
//
// Score (real between 0.0 and 1.0) written on stdout.
// Textual description of the result written on stderr.

#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <cstring>
#include <fstream>
#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <set>

using namespace std;

/**
 * @param p fraction of points awarded to the contestant.
 * @pararm m error message displayed to the contestant.
 */
void finish(double p, const string& m);

/**
 * The main checking function.
 * @param fin official input
 * @param foff official output
 * @param fout contestant's output
 */
void checker(ifstream& fin, ifstream& foff, ifstream& fout)
{
  const string WRONG_OUTPUT_FORMAT = "Incorrectly formatted output.";
  const string TEST_DATA_ERROR = "Error in official input or output.";
  const string WRONG = "Incorrect.";
  const string CORRECT = "Correct.";

  // Read official input
  int N;

  if (!(fin >> N)) finish(0, TEST_DATA_ERROR);
  vector<int> f(2*N, 0);
  for (int i = 0; i < 2*N; ++i) {
    if (!(fin >> f[i])) finish(0, TEST_DATA_ERROR);
    f[i]--;
  }
  
  // Read contestant's output
  vector<bool> inS(2*N, false);
  vector<bool> challengedByS(2*N, false);
  {
    int x;
    while (fout >> x) {
      --x;
      if (!(0 <= x && x < 2*N) || inS[x]) {
        finish(0, WRONG_OUTPUT_FORMAT);
      }
      inS[x] = true;
      challengedByS[f[x]] = true;
    }
  }
  if (!fout.eof()) finish(0, WRONG_OUTPUT_FORMAT);

  for (int i = 0; i < 2*N; ++i)
    if ((inS[i] && challengedByS[i]) || (!inS[i] && !challengedByS[i]))
      finish(0, WRONG);

  finish(1, CORRECT);
  // The function MUST terminate before this line via finish()!
}

void finish(double p, const string& m) {
  cout << p << endl;
  cout << m << endl;
  exit(0);
}

int main(int argc, char *argv[])
{
  assert(argc == 4);

  ifstream fin(argv[1]);
  ifstream foff(argv[2]);
  ifstream fout(argv[3]);

  assert(!fin.fail() && !fout.fail());
  checker(fin, foff, fout);
  assert(false); // checker must terminate via finish() before exiting!

  return 0;
}
