/* CERC 2011 
 * Solution to the problem Cards
 * Idea: find a solution to the set of at most five equations x = a_i mod m_i
 *       using a similar recursive approach as the Euclid's algorithm
 * Author: Josef Cibulka
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#define MAXPL 1000
#define CPERP 5
#define MAXN (CPERP*MAXPL)

// number of cards and of players
int n, pcnt;
// the card dealing permutation - perm[i]=place where card from place i is moved
int perm[MAXN];
// init[i] = initial positions of card i (inverse perm. to the input)
// only for the important cards - 1...CPERP
int init[CPERP];
	// in how many turns does the card return to its initial position
int cyclen[CPERP];
// when player has the card (up to 5 times within the card's cycle)
int whenphasc[MAXPL][CPERP][CPERP];
int whenphasccnt[MAXPL][CPERP];

// cnts[i] = number of cards in each dealing round i and cntssum[i] = sum from rounds 1..i-1
int cnts[CPERP] = {2,2,1,0,0};
int cntssum[CPERP] = {0,2,4,5,5};

// greatest common divisor -- Euler's algorithm
long long gcd(long long a, long long b)
{
	if(a==0 || a==b) return b;
	if(b==0) return a;
	if(a<b) return gcd(b%a, a);
	return gcd(a%b, b);
}

long long lcm(long long a, long long b)
{
	return (a/gcd(a,b))*b; // divide first to avoid overflow
}

// returns y such that ay=b mod m -- similar to Euler's algo
// assumes that a>=m
// if no solution, return -1
long long subsolve(long long a, long long b, long long m)
{
	long long mnew, bnew, subres, ret;
	assert(a>=m);
	assert(gcd(a,m)==1);
	if(b==0) return 0;
	mnew = a%m;
	assert(mnew != 0);
	bnew = mnew - 1 - ((b-1)%mnew);
	subres = subsolve(m, bnew, mnew);
	// now subres*m = bnew mod mnew
	// that is subres*m + b = 0 mod mnew   because bnew = -b mod mnew
	// then ((subres*m+b)/mnew) * a = (a mod m)*((subres*m+b)/mnew)	mod m = subres*m+b mod m = b mod m
	ret = (subres*m+b)/mnew;
	assert(ret*mnew == subres*m+b);
	return ret%m;
}


// solves the set of equations and returns smallest positive solution 
// (if no solution, return -1):
// x = pert[0] mod per[0]
// ...
// x = pert[cnt-1] mod per[cnt-1]
long long solve(int cnt, int *per, int *pert)
{
	int i;
	long long val, valmodg, val1, val2, mod, mod1, mod2, g, tmp;
	
	if(cnt==0) return 1;
	val = pert[0];
	mod = per[0];

	for(i=1;i<cnt;i++)
	{
		// solve x = val mod mod   and  x = pert[i] mod per[i] and store the result to val
		// first if g = gcd(mod, per[i]) > 1 we check if there is a solution and if there is, divide by g
		g = gcd(mod, per[i]);
 		if((val%g) != (pert[i]%g)) return -1;
		valmodg = val%g;
		val1 = val/g;
		val2 = pert[i]/g;
		mod1 = mod/g;
		mod2 = per[i]/g;
		// transform to form acceptable for subsolve
		if(mod1<mod2) 
		{ 
			tmp = val1; val1 = val2; val2 = tmp; 
			tmp = mod1; mod1 = mod2; mod2 = tmp; 
		}
		if(val2>=val1) tmp = val2-val1;
		else 
		{ 
			assert(val2 + ((val1-val2-1)/mod2 + 1)*mod2>=val1); 
			tmp = val2 + ((val1-val2-1)/mod2 + 1)*mod2 - val1;
		}
		assert(tmp<mod2);
		// subsolve will solve mod1*x = val2-val1 mod mod2 
		val = g * (mod1 * subsolve(mod1, tmp, mod2) + val1) + valmodg;
		mod = mod1*mod2*g;
	}
	if(val==0) return mod;
	else return val;
}

int sbtcnt;
int sbtlen[CPERP];
int sbttime[CPERP];
// if some cards has more moments when it appears in the player p's hand,
// try all combinations of these times and send them to solve
long long solvebt(int p, int dep)
{
	int i;
	long long tmp, ret;

	if(dep==0) sbtcnt = 0;
	if(dep == CPERP) return solve(sbtcnt, sbtlen, sbttime);
	if(whenphasccnt[p][dep]==0) return -1;
	if(whenphasccnt[p][dep]==-1 || cyclen[dep]==1) return solvebt(p,dep+1);
	assert(whenphasccnt[p][dep]>=1);
	ret = -1;
	for(i=0; i<whenphasccnt[p][dep]; i++)
	{
		sbtlen[sbtcnt] = cyclen[dep];
		sbttime[sbtcnt++] = whenphasc[p][dep][i];
		tmp = solvebt(p,dep+1);
		if(tmp>-1 && (ret==-1 || tmp<ret)) ret = tmp;
		sbtcnt--;
	}
	return ret;
}


int main(void)
{
	int p,ii,jj,i,j,k,tmp,tmp2,len;
	long long tmpll, winround;
	int winner;

	while(1)
	{
		scanf(" %d ", &pcnt);
		if(pcnt==0) break;
		n = 5*pcnt;
		assert(n<=MAXN);

		for(i=0;i<n;i++)
		{
			scanf(" %d ", &tmp);
			tmp--;
			if(tmp<CPERP) init[tmp] = i;
		}
			
		// find the dealing permutation
		tmp=tmp2=0;
		for(i=0;i<5;i++)
		{
			for(j=0;j<pcnt;j++)
			{
				for(k=0;k<cnts[i];k++) 
				{
					tmp2 = 5*j+cntssum[i]+k;
					perm[tmp] = tmp2;
					tmp++;
				}
			}
		}

		// find cycle lengths and also for each player and each important card 
		// all the times when the player has it
		for(i=0;i<pcnt;i++) for(j=0;j<CPERP;j++) whenphasccnt[i][j]=0;
		for(i=0;i<CPERP;i++) 
		{
			tmp = init[i];
			len = 0;
			while(len==0 || tmp!=init[i]) 
			{ 
				tmp2 = tmp/CPERP;
				whenphasc[tmp2][i][whenphasccnt[tmp2][i]++] = len;
				tmp = perm[tmp]; 
				len++; 
			}
			cyclen[i] = len;
		}

		winround = winner = -1;
		// for each player find out if and when will he win
		for(p=0;p<pcnt;p++)
		{
			// preprocess -- if two card cycles have the same length, then they must meet 
			// in the player's hand at the same time
			for(i=0;i<CPERP;i++) for(j=i+1;j<CPERP;j++) if(cyclen[i]!=-1 && cyclen[i]==cyclen[j])
			{
				for(ii=0;ii<whenphasccnt[p][i];ii++) 
				{
					for(jj=0;jj<whenphasccnt[p][j];jj++) if(whenphasc[p][i][ii] == whenphasc[p][j][jj]) break;
					if(jj==whenphasccnt[p][j]) // remove the ii-th occurrence of card i
					{
						for(k=ii;k<whenphasccnt[p][i]-1;k++) whenphasc[p][i][k] = whenphasc[p][i][k+1];
						whenphasccnt[p][i]--;
						ii--;
					}
				}
				whenphasccnt[p][j]=-1; // we've just merged the restrictions from card j with those from card i
			}
		
			tmpll = solvebt(p, 0);
			if(tmpll>-1) 
			{
				if(winner==-1 || tmpll<winround) { winner = p+1; winround = tmpll; }
			}
		}
		
		if(winner==-1) printf("Neverending game.\n");
		else printf("Player %d wins game number %lld.\n", winner, winround); 
	}
	return 0;
}
